"use client"

import { useState } from "react"
import { useAuth } from "@/lib/auth-context"
import { useChat } from "@/lib/chat-context"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { Search, MoreVertical, MessageSquarePlus, LogOut, Users, User } from "lucide-react"
import { formatDistanceToNow } from "date-fns"
import { tr } from "date-fns/locale"
import { ContactList } from "./contact-list"
import { ProfileModal } from "./profile-modal"

export function ChatSidebar() {
  const { user, logout } = useAuth()
  const { chats, messages, activeChat, setActiveChat, getContactInfo } = useChat()
  const [searchQuery, setSearchQuery] = useState("")
  const [showContacts, setShowContacts] = useState(false)
  const [showProfile, setShowProfile] = useState(false)

  const getLastMessage = (chatId: string) => {
    const chatMessages = messages.filter((m) => m.chatId === chatId)
    return chatMessages[chatMessages.length - 1]
  }

  const filteredChats = chats.filter((chat) => {
    const contact = getContactInfo(chat.participants)
    return contact?.displayName.toLowerCase().includes(searchQuery.toLowerCase())
  })

  if (showContacts) {
    return <ContactList onBack={() => setShowContacts(false)} />
  }

  return (
    <div className="w-full md:w-[400px] h-full flex flex-col bg-card border-r border-border">
      <div className="h-16 px-4 flex items-center justify-between bg-secondary">
        <div className="flex items-center gap-3">
          <Avatar
            className="w-10 h-10 cursor-pointer hover:opacity-80 transition-opacity"
            onClick={() => setShowProfile(true)}
          >
            <AvatarFallback className="bg-primary text-primary-foreground">
              {user?.displayName?.charAt(0).toUpperCase()}
            </AvatarFallback>
          </Avatar>
          <span className="font-medium text-foreground">{user?.displayName}</span>
        </div>
        <div className="flex items-center gap-2">
          <Button
            variant="ghost"
            size="icon"
            className="text-muted-foreground hover:text-foreground hover:bg-popover"
            onClick={() => setShowContacts(true)}
          >
            <MessageSquarePlus className="w-5 h-5" />
          </Button>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                size="icon"
                className="text-muted-foreground hover:text-foreground hover:bg-popover"
              >
                <MoreVertical className="w-5 h-5" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="bg-popover border-border">
              <DropdownMenuItem
                onClick={() => setShowProfile(true)}
                className="text-foreground hover:bg-secondary cursor-pointer"
              >
                <User className="w-4 h-4 mr-2" />
                Profil
              </DropdownMenuItem>
              <DropdownMenuItem
                onClick={() => setShowContacts(true)}
                className="text-foreground hover:bg-secondary cursor-pointer"
              >
                <Users className="w-4 h-4 mr-2" />
                Kişiler
              </DropdownMenuItem>
              <DropdownMenuSeparator className="bg-border" />
              <DropdownMenuItem onClick={logout} className="text-destructive hover:bg-secondary cursor-pointer">
                <LogOut className="w-4 h-4 mr-2" />
                Çıkış Yap
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>

      <div className="p-2 bg-card">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
          <Input
            placeholder="Sohbet ara veya yeni sohbet başlat"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10 bg-secondary border-0 text-foreground placeholder:text-muted-foreground focus-visible:ring-0"
          />
        </div>
      </div>

      <div className="flex-1 overflow-y-auto">
        {filteredChats.length === 0 ? (
          <div className="flex flex-col items-center justify-center h-full text-muted-foreground p-4">
            <MessageSquarePlus className="w-12 h-12 mb-4" />
            <p className="text-center">Henüz sohbet yok</p>
            <Button variant="link" className="text-primary" onClick={() => setShowContacts(true)}>
              Yeni sohbet başlat
            </Button>
          </div>
        ) : (
          filteredChats.map((chat) => {
            const contact = getContactInfo(chat.participants)
            const lastMessage = getLastMessage(chat.id)
            const isActive = activeChat?.id === chat.id

            return (
              <div
                key={chat.id}
                onClick={() => setActiveChat(chat)}
                className={`flex items-center gap-3 px-4 py-3 cursor-pointer hover:bg-secondary transition-colors ${
                  isActive ? "bg-secondary" : ""
                }`}
              >
                <Avatar className="w-12 h-12">
                  <AvatarFallback className="bg-accent text-foreground">
                    {contact?.displayName?.charAt(0).toUpperCase()}
                  </AvatarFallback>
                </Avatar>
                <div className="flex-1 min-w-0">
                  <div className="flex items-center justify-between">
                    <span className="font-medium text-foreground truncate">{contact?.displayName}</span>
                    {lastMessage && (
                      <span className="text-xs text-muted-foreground">
                        {formatDistanceToNow(new Date(lastMessage.timestamp), {
                          addSuffix: false,
                          locale: tr,
                        })}
                      </span>
                    )}
                  </div>
                  <div className="flex items-center justify-between">
                    <p className="text-sm text-muted-foreground truncate">
                      {lastMessage?.content || "Sohbete başlayın"}
                    </p>
                    {chat.unreadCount > 0 && (
                      <span className="bg-primary text-primary-foreground text-xs rounded-full px-2 py-0.5">
                        {chat.unreadCount}
                      </span>
                    )}
                  </div>
                </div>
              </div>
            )
          })
        )}
      </div>

      <ProfileModal open={showProfile} onOpenChange={setShowProfile} />
    </div>
  )
}
