"use client"

import { useState } from "react"
import { useAuth } from "@/lib/auth-context"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { Label } from "@/components/ui/label"
import { Pencil, Check, X } from "lucide-react"

interface ProfileModalProps {
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ProfileModal({ open, onOpenChange }: ProfileModalProps) {
  const { user, updateProfile } = useAuth()
  const [isEditingName, setIsEditingName] = useState(false)
  const [isEditingStatus, setIsEditingStatus] = useState(false)
  const [displayName, setDisplayName] = useState(user?.displayName || "")
  const [status, setStatus] = useState(user?.status || "")

  const handleSaveName = () => {
    if (displayName.trim()) {
      updateProfile({ displayName: displayName.trim() })
      setIsEditingName(false)
    }
  }

  const handleSaveStatus = () => {
    updateProfile({ status: status.trim() || "Merhaba! WhatsApp kullanıyorum." })
    setIsEditingStatus(false)
  }

  const handleCancelName = () => {
    setDisplayName(user?.displayName || "")
    setIsEditingName(false)
  }

  const handleCancelStatus = () => {
    setStatus(user?.status || "")
    setIsEditingStatus(false)
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="bg-card border-border text-foreground sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Profil</DialogTitle>
        </DialogHeader>
        <div className="flex flex-col items-center py-4">
          <Avatar className="w-32 h-32 mb-4">
            <AvatarFallback className="bg-primary text-primary-foreground text-4xl">
              {user?.displayName?.charAt(0).toUpperCase()}
            </AvatarFallback>
          </Avatar>

          <div className="w-full space-y-4">
            <div className="space-y-2">
              <Label className="text-primary text-xs">Adınız</Label>
              {isEditingName ? (
                <div className="flex items-center gap-2">
                  <Input
                    value={displayName}
                    onChange={(e) => setDisplayName(e.target.value)}
                    className="bg-secondary border-0 text-foreground focus-visible:ring-0"
                    autoFocus
                  />
                  <Button size="icon" variant="ghost" onClick={handleSaveName} className="text-primary shrink-0">
                    <Check className="w-5 h-5" />
                  </Button>
                  <Button
                    size="icon"
                    variant="ghost"
                    onClick={handleCancelName}
                    className="text-muted-foreground shrink-0"
                  >
                    <X className="w-5 h-5" />
                  </Button>
                </div>
              ) : (
                <div
                  className="flex items-center justify-between py-2 cursor-pointer hover:bg-secondary px-2 rounded"
                  onClick={() => setIsEditingName(true)}
                >
                  <span>{user?.displayName}</span>
                  <Pencil className="w-4 h-4 text-muted-foreground" />
                </div>
              )}
            </div>

            <div className="space-y-2">
              <Label className="text-primary text-xs">Durum</Label>
              {isEditingStatus ? (
                <div className="flex items-center gap-2">
                  <Input
                    value={status}
                    onChange={(e) => setStatus(e.target.value)}
                    className="bg-secondary border-0 text-foreground focus-visible:ring-0"
                    autoFocus
                    maxLength={139}
                  />
                  <Button size="icon" variant="ghost" onClick={handleSaveStatus} className="text-primary shrink-0">
                    <Check className="w-5 h-5" />
                  </Button>
                  <Button
                    size="icon"
                    variant="ghost"
                    onClick={handleCancelStatus}
                    className="text-muted-foreground shrink-0"
                  >
                    <X className="w-5 h-5" />
                  </Button>
                </div>
              ) : (
                <div
                  className="flex items-center justify-between py-2 cursor-pointer hover:bg-secondary px-2 rounded"
                  onClick={() => setIsEditingStatus(true)}
                >
                  <span className="text-muted-foreground">{user?.status}</span>
                  <Pencil className="w-4 h-4 text-muted-foreground" />
                </div>
              )}
            </div>

            <div className="space-y-2">
              <Label className="text-primary text-xs">Kullanıcı Adı</Label>
              <div className="py-2 px-2 text-muted-foreground">@{user?.username}</div>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  )
}
