"use client"

import { createContext, useContext, useState, useEffect, type ReactNode } from "react"
import type { User, Message, Chat } from "./types"
import { useAuth } from "./auth-context"

interface ChatContextType {
  chats: Chat[]
  messages: Message[]
  contacts: User[]
  activeChat: Chat | null
  setActiveChat: (chat: Chat | null) => void
  sendMessage: (content: string) => void
  startChat: (userId: string) => void
  getContactInfo: (participantIds: string[]) => User | undefined
}

const ChatContext = createContext<ChatContextType | undefined>(undefined)

const demoUsers: User[] = [
  {
    id: "demo-1",
    username: "ahmet",
    password: "123456",
    displayName: "Ahmet Yılmaz",
    status: "İş toplantısındayım",
    lastSeen: new Date(Date.now() - 1000 * 60 * 5),
  },
  {
    id: "demo-2",
    username: "ayse",
    password: "123456",
    displayName: "Ayşe Demir",
    status: "Merhaba! WhatsApp kullanıyorum.",
    lastSeen: new Date(Date.now() - 1000 * 60 * 30),
  },
  {
    id: "demo-3",
    username: "mehmet",
    password: "123456",
    displayName: "Mehmet Kaya",
    status: "Sadece acil aramalar",
    lastSeen: new Date(Date.now() - 1000 * 60 * 60 * 2),
  },
  {
    id: "demo-4",
    username: "fatma",
    password: "123456",
    displayName: "Fatma Öztürk",
    status: "Müsait değilim",
    lastSeen: new Date(),
  },
  {
    id: "demo-5",
    username: "ali",
    password: "123456",
    displayName: "Ali Çelik",
    status: "🎵 Müzik dinliyorum",
    lastSeen: new Date(Date.now() - 1000 * 60 * 15),
  },
]

export function ChatProvider({ children }: { children: ReactNode }) {
  const { user } = useAuth()
  const [chats, setChats] = useState<Chat[]>([])
  const [messages, setMessages] = useState<Message[]>([])
  const [contacts, setContacts] = useState<User[]>([])
  const [activeChat, setActiveChat] = useState<Chat | null>(null)

  useEffect(() => {
    if (user) {
      const existingUsers = localStorage.getItem("users")
      const users: User[] = existingUsers ? JSON.parse(existingUsers) : []

      const allContacts = [...demoUsers]
      users.forEach((u) => {
        if (!allContacts.find((c) => c.id === u.id) && u.id !== user.id) {
          allContacts.push(u)
        }
      })
      setContacts(allContacts.filter((c) => c.id !== user.id))

      const storedChats = localStorage.getItem(`chats-${user.id}`)
      if (storedChats) {
        setChats(JSON.parse(storedChats))
      } else {
        const demoChats: Chat[] = demoUsers.slice(0, 3).map((contact) => ({
          id: `chat-${user.id}-${contact.id}`,
          participants: [user.id, contact.id],
          unreadCount: Math.floor(Math.random() * 5),
        }))
        setChats(demoChats)
        localStorage.setItem(`chats-${user.id}`, JSON.stringify(demoChats))
      }

      const storedMessages = localStorage.getItem(`messages-${user.id}`)
      if (storedMessages) {
        setMessages(JSON.parse(storedMessages))
      } else {
        const demoMessages: Message[] = []
        demoUsers.slice(0, 3).forEach((contact) => {
          const chatId = `chat-${user.id}-${contact.id}`
          demoMessages.push(
            {
              id: crypto.randomUUID(),
              chatId,
              senderId: contact.id,
              content: "Merhaba! Nasılsın?",
              timestamp: new Date(Date.now() - 1000 * 60 * 60),
              status: "read",
            },
            {
              id: crypto.randomUUID(),
              chatId,
              senderId: user.id,
              content: "İyiyim, sen nasılsın?",
              timestamp: new Date(Date.now() - 1000 * 60 * 55),
              status: "read",
            },
            {
              id: crypto.randomUUID(),
              chatId,
              senderId: contact.id,
              content: "Ben de iyiyim, teşekkürler!",
              timestamp: new Date(Date.now() - 1000 * 60 * 50),
              status: "read",
            },
          )
        })
        setMessages(demoMessages)
        localStorage.setItem(`messages-${user.id}`, JSON.stringify(demoMessages))
      }
    }
  }, [user])

  const getContactInfo = (participantIds: string[]): User | undefined => {
    const contactId = participantIds.find((id) => id !== user?.id)
    return contacts.find((c) => c.id === contactId)
  }

  const sendMessage = (content: string) => {
    if (!activeChat || !user) return

    const newMessage: Message = {
      id: crypto.randomUUID(),
      chatId: activeChat.id,
      senderId: user.id,
      content,
      timestamp: new Date(),
      status: "sent",
    }

    const updatedMessages = [...messages, newMessage]
    setMessages(updatedMessages)
    localStorage.setItem(`messages-${user.id}`, JSON.stringify(updatedMessages))

    const updatedChats = chats.map((chat) => (chat.id === activeChat.id ? { ...chat, lastMessage: newMessage } : chat))
    setChats(updatedChats)
    localStorage.setItem(`chats-${user.id}`, JSON.stringify(updatedChats))

    setTimeout(
      () => {
        const contact = getContactInfo(activeChat.participants)
        if (contact) {
          const replies = [
            "Anladım 👍",
            "Tamam, teşekkürler!",
            "Peki, görüşürüz!",
            "Harika!",
            "Biraz sonra yazarım",
            "👋",
            "OK",
          ]
          const replyMessage: Message = {
            id: crypto.randomUUID(),
            chatId: activeChat.id,
            senderId: contact.id,
            content: replies[Math.floor(Math.random() * replies.length)],
            timestamp: new Date(),
            status: "delivered",
          }
          setMessages((prev) => {
            const updated = [...prev, replyMessage]
            localStorage.setItem(`messages-${user.id}`, JSON.stringify(updated))
            return updated
          })
        }
      },
      1000 + Math.random() * 2000,
    )
  }

  const startChat = (userId: string) => {
    if (!user) return

    const existingChat = chats.find((chat) => chat.participants.includes(userId) && chat.participants.includes(user.id))

    if (existingChat) {
      setActiveChat(existingChat)
    } else {
      const newChat: Chat = {
        id: `chat-${user.id}-${userId}`,
        participants: [user.id, userId],
        unreadCount: 0,
      }
      const updatedChats = [newChat, ...chats]
      setChats(updatedChats)
      localStorage.setItem(`chats-${user.id}`, JSON.stringify(updatedChats))
      setActiveChat(newChat)
    }
  }

  return (
    <ChatContext.Provider
      value={{
        chats,
        messages,
        contacts,
        activeChat,
        setActiveChat,
        sendMessage,
        startChat,
        getContactInfo,
      }}
    >
      {children}
    </ChatContext.Provider>
  )
}

export function useChat() {
  const context = useContext(ChatContext)
  if (context === undefined) {
    throw new Error("useChat must be used within a ChatProvider")
  }
  return context
}
